<?php

namespace App\Http\Controllers;

use App\Config;
use App\Gst;
use App\Movie;
use App\Package;
use App\PpvPurchase;
use App\Season;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;

class PhonePayController extends Controller
{
    public function initiatePayment(Request $request)
    {
        Log::info('Payment initiation started', [
            'user_id' => Auth::id(),
            'amount' => $request->amount,
            'plan_id' => $request->plan_id,
            'type' => $request->type
        ]);

        $merchantId = env('PHONEPE_MERCHANT_ID');
        $apiKey = env('PHONEPE_SALT_KEY');
        $redirect = rtrim(env('PHONEPE_REDIRECT_URL'), '/');
        $merchantTransactionId = 'MT' . uniqid();

        // Save context for use in callback, updated to match Razorpay
        session([
            'phonepe_transaction' => [
                'merchant_transaction_id' => $merchantTransactionId,
                'amount' => $request->amount,
                'plan_id' => $request->plan_id,
                'type' => $request->type,
                'movie_id' => $request->movie_id,
                'season_id' => $request->series_id,
            ]
        ]);

        $redirectUrl = "{$redirect}/redirect-url?mtid={$merchantTransactionId}";

        $transactionData = [
            'merchantId' => $merchantId,
            'merchantTransactionId' => $merchantTransactionId,
            'merchantUserId' => 'MUID' . Auth::id(),
            'amount' => $request->amount * 100, // Convert to paise
            'redirectUrl' => $redirectUrl,
            'redirectMode' => 'GET',
            'callbackUrl' => "{$redirect}/callback-url",
            'mobileNumber' => Auth::user()->mobile ?? '',
            'paymentInstrument' => [
                'type' => 'PAY_PAGE',
            ],
        ];

        $payloadMain = base64_encode(json_encode($transactionData));
        $saltIndex = 1;
        $payload = "{$payloadMain}/pg/v1/pay{$apiKey}";
        $sha256 = hash("sha256", $payload);
        $finalXHeader = "{$sha256}###{$saltIndex}";

        try {
            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'X-VERIFY' => $finalXHeader,
                'accept' => 'application/json',
            ])->post('https://api.phonepe.com/apis/hermes/pg/v1/pay', [
                'request' => $payloadMain
            ]);

            Log::info('PhonePe API response received', [
                'status_code' => $response->status(),
                'response_body' => $response->json()
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $payUrl = $data['data']['instrumentResponse']['redirectInfo']['url'] ?? null;
                if ($payUrl) return redirect()->away($payUrl);
            }

            return back()->withErrors(['error' => __('Payment initiation failed')]);
        } catch (\Exception $e) {
            Log::error('Payment API Exception', [
                'message' => $e->getMessage(),
                'transaction_id' => $merchantTransactionId
            ]);
            return back()->withErrors(['error' => __('An unexpected error occurred')]);
        }
    }

    public function handleRedirect(Request $request)
    {
        Log::info('Redirect URL hit', [
            'all_parameters' => $request->all(),
            'mtid' => $request->query('mtid')
        ]);

        try {
            $context = session('phonepe_transaction');
            $mtid = $request->query('mtid');

            if (!$context || !isset($context['merchant_transaction_id']) || $context['merchant_transaction_id'] !== $mtid) {
                Log::error('Transaction mismatch or null context', [
                    'stored' => $context['merchant_transaction_id'] ?? 'null',
                    'received' => $mtid ?? 'null'
                ]);
                return redirect('/')->with('error', __('Invalid transaction'));
            }

            $type = $context['type'] ?? null;
            $amount = $context['amount'] ?? null;
            $gst = Gst::first();
            $config = Config::findOrFail(1);
            $base_currency = $config->currency_code ?? 'INR'; // Fallback to INR if null
            $auth = Auth::user();

            if (!$type || !$amount || !is_numeric($amount)) {
                Log::error('Invalid or null type/amount', [
                    'type' => $type,
                    'amount' => $amount
                ]);
                return redirect('/')->with('error', __('Invalid payment details'));
            }

            if ($type === 'subscription') {
                Log::info('Processing subscription payment', ['plan_id' => $context['plan_id']]);
                $plan = Package::findOrFail($context['plan_id']);
                $checkout = new SubscriptionController;
                $result = $checkout->subscribe(
                    $mtid,
                    'phonepe',
                    $plan->id,
                    1,
                    $amount
                );
                session()->forget('phonepe_transaction');
                return redirect('/')->with('success', __('Subscription successful'));
            }

            if ($type === 'rental') {
                $movie_id = $context['movie_id'] ?? null;
                if (!$movie_id) {
                    Log::error('Null movie_id for rental', ['context' => $context]);
                    return redirect('/')->with('error', __('Invalid movie ID'));
                }

                $movie = Movie::findOrFail($movie_id);
                if (!$movie->hours || !$movie->slug) {
                    Log::error('Null movie hours or slug', [
                        'movie_id' => $movie_id,
                        'hours' => $movie->hours,
                        'slug' => $movie->slug
                    ]);
                    return redirect('/')->with('error', __('Invalid movie data'));
                }

                $converted_price = Session::has('current_currency')
                    ? round(currency($movie->price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                    : round($movie->price, 2);

                $converted_offer_price = null;
                if (isset($movie->offer_price) && $movie->offer_price > 0) {
                    $converted_offer_price = Session::has('current_currency')
                        ? round(currency($movie->offer_price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                        : round($movie->offer_price, 2);
                }

                $base_price = $converted_offer_price ?? $converted_price;

                $gst_amount = 0;
                $total_with_gst = $base_price;
                if ($gst && $gst->gst_enable && isset($gst->gst_per) && is_numeric($gst->gst_per)) {
                    $gst_amount = ($base_price * $gst->gst_per) / 100;
                    $total_with_gst = $base_price + $gst_amount;
                } else {
                    Log::info('GST disabled or null', [
                        'gst_enable' => $gst->gst_enable ?? 'null',
                        'gst_per' => $gst->gst_per ?? 'null'
                    ]);
                }

                Log::info('Movie rental payment details', [
                    'movie_id' => $movie->id,
                    'movie_title' => $movie->title,
                    'base_price' => $base_price,
                    'gst_percentage' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                    'gst_amount' => $gst_amount,
                    'total_with_gst' => $total_with_gst,
                    'payment_amount' => $amount,
                    'converted_price' => $converted_price,
                    'converted_offer_price' => $converted_offer_price,
                    'current_currency' => Session::get('current_currency') ?? $base_currency
                ]);

                if (abs($amount - $total_with_gst) > 0.01) {
                    Log::error('Payment amount mismatch for movie rental', [
                        'expected' => $total_with_gst,
                        'received' => $amount,
                        'difference' => $total_with_gst - $amount
                    ]);

                    if (abs($amount - $total_with_gst) > 1.00) {
                        return redirect()->back()->with('error', __('Payment amount mismatch. Please try again.'));
                    }
                }

                $purchase = PpvPurchase::create([
                    'user_id' => $auth->id,
                    'movie_id' => $movie->id,
                    'payment_id' => $mtid,
                    'gst' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                    'price' => $converted_price,
                    'offer_price' => $converted_offer_price ?? $converted_price,
                    'hours' => $movie->hours,
                    'expires_at' => now()->addHours($movie->hours),
                    'type' => 'movie',
                    'payment_type' => 'PhonePe',
                    'status' => 1,
                ]);

                Log::info('PPV purchase created for movie', ['purchase_id' => $purchase->id]);
                session()->forget('phonepe_transaction');

                $route = getSubscription()->getData()->subscribed
                    ? 'movie/detail/'
                    : 'movie/guest/detail/';
                return redirect()->to($route . $movie->slug)->with('success', __('Movie purchase successful!'));
            }

            if ($type === 'rental_series') {
                $season_id = $context['season_id'] ?? null;
                if (!$season_id) {
                    Log::error('Null season_id for rental_series', ['context' => $context]);
                    return redirect('/')->with('error', __('Invalid season ID'));
                }

                $season = Season::findOrFail($season_id);
                if (!$season->hours || !$season->season_slug) {
                    Log::error('Null season hours or slug', [
                        'season_id' => $season_id,
                        'hours' => $season->hours,
                        'season_slug' => $season->season_slug
                    ]);
                    return redirect('/')->with('error', __('Invalid season data'));
                }

                $converted_price = Session::has('current_currency')
                    ? round(currency($season->price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                    : round($season->price, 2);

                $converted_offer_price = null;
                if (isset($season->offer_price) && $season->offer_price > 0) {
                    $converted_offer_price = Session::has('current_currency')
                        ? round(currency($season->offer_price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                        : round($season->offer_price, 2);
                }

                $base_price = $converted_offer_price ?? $converted_price;

                $gst_amount = 0;
                $total_with_gst = $base_price;
                if ($gst && $gst->gst_enable && isset($gst->gst_per) && is_numeric($gst->gst_per)) {
                    $gst_amount = ($base_price * $gst->gst_per) / 100;
                    $total_with_gst = $base_price + $gst_amount;
                } else {
                    Log::info('GST disabled or null', [
                        'gst_enable' => $gst->gst_enable ?? 'null',
                        'gst_per' => $gst->gst_per ?? 'null'
                    ]);
                }

                Log::info('Season rental payment details', [
                    'season_id' => $season->id,
                    'season_title' => 'Season ' . $season->season_no,
                    'base_price' => $base_price,
                    'gst_percentage' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                    'gst_amount' => $gst_amount,
                    'total_with_gst' => $total_with_gst,
                    'payment_amount' => $amount,
                    'converted_price' => $converted_price,
                    'converted_offer_price' => $converted_offer_price,
                    'current_currency' => Session::get('current_currency') ?? $base_currency
                ]);

                if (abs($amount - $total_with_gst) > 0.01) {
                    Log::error('Payment amount mismatch for season rental', [
                        'expected' => $total_with_gst,
                        'received' => $amount,
                        'difference' => $total_with_gst - $amount
                    ]);

                    if (abs($amount - $total_with_gst) > 1.00) {
                        return redirect()->back()->with('error', __('Payment amount mismatch. Please try again.'));
                    }
                }

                $purchase = PpvPurchase::create([
                    'user_id' => $auth->id,
                    'season_id' => $season->id,
                    'payment_id' => $mtid,
                    'gst' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                    'price' => $converted_price,
                    'offer_price' => $converted_offer_price ?? $converted_price,
                    'hours' => $season->hours,
                    'expires_at' => now()->addHours($season->hours),
                    'type' => 'series',
                    'payment_type' => 'PhonePe',
                    'status' => 1,
                ]);

                Log::info('PPV purchase created for season', ['purchase_id' => $purchase->id]);
                session()->forget('phonepe_transaction');

                $route = getSubscription()->getData()->subscribed
                    ? 'show/detail/'
                    : 'show/guest/detail/';
                return redirect()->to($route . $season->season_slug)->with('success', __('Season rental successful!'));
            }

            Log::error('Invalid payment type', ['type' => $type]);
            return redirect('/')->with('error', __('Invalid payment type'));
        } catch (\Exception $e) {
            Log::error('Redirect handling exception', [
                'message' => $e->getMessage(),
                'request_data' => $request->all()
            ]);
            return redirect('/')->with('error', __('Error processing payment'));
        }
    }
}